#include <assert.h>

#include <string>

#include "lib/util.h"
#include "watch_maildirs.h"
#include "watcher_partial.h"
#include "watcher_maildir.h"

using std::string;

watcher_maildir::watcher_maildir(struct inotify_state* state, const string& maildir)
	: maildir(maildir), imap_name(maildir_to_imap(maildir)), state(state)
{
	string full_maildir(mkfilename(state->base_path, maildir));
	string maildir_cur(mkfilename(full_maildir, "cur/"));;
	string maildir_new(mkfilename(full_maildir, "new/"));

	wd_maildir = start_watch(full_maildir.c_str(), m_notify);
	wd_cur = start_watch(maildir_cur.c_str(), cn_notify);
	wd_new = start_watch(maildir_new.c_str(), cn_notify);
}

watcher_maildir::~watcher_maildir()
{
	if (wd_maildir >= 0)
		stop_watch(wd_maildir);
	if (wd_cur >= 0)
		stop_watch(wd_cur);
	if (wd_new >= 0)
		stop_watch(wd_new);
}


void watcher_maildir::process_event(const struct inotify_event& event)
{
	if (event.wd == wd_cur || event.wd == wd_new)
		send_maildir_modified(imap_name.c_str());
	else
	{
		assert(event.wd == wd_maildir);
		if (!(event.mask & IN_ISDIR))
			return;
		if (!event.len || (strcmp(event.name, "cur") && strcmp(event.name, "new")))
			return;

		send_maildir_modified(imap_name.c_str());

		stop_watch(wd_maildir);
		if (strcmp(event.name, "cur"))
			stop_watch(wd_cur);
		if (strcmp(event.name, "new"))
			stop_watch(wd_new);
		wd_maildir = wd_cur = wd_new = -1;

		new watcher_partial(state, maildir);

		delete this;
		return;
	}
}

int watcher_maildir::start_watch(const char* dir, uint32_t notify)
{
	int wd = inotify_add_watch(state->inotify_fd, dir, notify);
	die_if(wd < 0, "inotify_add_watch(\"%s\")", dir);
	state->add_imap_watcher(wd, imap_name, this);
	return wd;
}

void watcher_maildir::stop_watch(int wd)
{
	int r = inotify_rm_watch(state->inotify_fd, wd);
	die_if(r < 0, "inotify_rm_watch(\"%s\")\n", imap_name.c_str());
	state->remove_imap_watcher(wd, imap_name);
}
